//	demo.cpp  -  an example using the MetaKit C++ persistence library
//
//	This is a part of the MetaKit library.
//	Copyright (c) 1996 Meta Four Software.
//	All rights reserved.
/////////////////////////////////////////////////////////////////////////////
//
//  This code demonstrates:
//
//      - Creating a persistent view and adding two data rows to it.
//      - Adding a third data row using MetaKit's operator shorthands.
//      - Adding an additional property without losing the existing data.
//      - Storing an additional view in the data file later on.
//      - Inserting a new record into one of the views in the datafile.
//      - Real persistence, the data file will grow each time this is run.
//      - The very powerful effects of on-the-fly data file restructuring.
//
/////////////////////////////////////////////////////////////////////////////

#include "m4kit.h"

#include <stdio.h>

int main()
{
        // These properties could just as well have been declared globally.
    c4_StringProp pName ("name");
    c4_StringProp pCountry ("country");

        // Note: it is extremely important to watch the lifetime of views vs.
        // storage objects.  If a view outlives its storage object, all data
        // will be loaded when the storage object is destroyed.  So much for
        // on-demand loading.  To avoid this, delete/clear all views before
        // the storage object is destroyed.  In this stack-based example, it
        // can easily be enforced by constructing the storage objects first.
    c4_Storage storage ("myfile.dat", true);

    // There are two ways to make views persistent: make sure the storage
	// structure is appropriately defined - then use a view based on it,
	// or build the entire structure first and then attach it to a storage
	// object (causes a deep copy, requiring twice the amount of memory).
	
	// The first option is preferred when adding large a amount of data:
	//		c4_View vAddress = storage.Define("address[name:S,country:S]");
    // The second approach (used here) has the advantage that we do not need
	// to describe the data structure, since Attach will determine it for us. 
    
		// Start with an empty view, not associated with any storage.
	c4_View vAddress;

		// Let's add two rows of data to the view.
    c4_Row row;

    pName (row) = "John Williams";
    pCountry (row) = "UK";
    vAddress.Add(row);

    pName (row) = "Paco Pena";
    pCountry (row) = "Spain";
    vAddress.Add(row);

		// Define the storage structure, and store the contents of vAddress.
	vAddress = storage.Store("address", vAddress);

        // A simple check to prove that the data is in the view.
    c4_String s1 = pName (vAddress[1]);
    c4_String s2 = pCountry (vAddress[1]);
    printf("The country of %s is: %s\n",
             (const char*) s1, (const char*) s2);

        // This saves the data to file.
    storage.Commit();   // Data file now contains 2 addresses.

        // A very compact notation to create and add a third row.
    vAddress.Add(pName ["Julien Coco"] + pCountry ["Netherlands"]);

    storage.Commit();   // Data file now contains 3 addresses.

        // Add a third property to the address view ("on-the-fly").
    vAddress = storage.GetAs("address[name:S,country:S,age:I]");

        // Set the new age property in one of the exisiting addresses.
    c4_IntProp pAge ("age");
    pAge (vAddress[1]) = 44;

    storage.Commit();   // Data file now contains 3 addresses with age field.

        // Add a second view to the data file, leaving the first view intact.
    c4_View vInfo = storage.GetAs("info[version:I]");

        // Add some data, a single integer in this case.
    c4_IntProp pVersion ("version");
    vInfo.Add(pVersion [100]);

    storage.Commit();   // Data file now contains 3 addresses and 1 info rec.

        // Insert a row into the address view.  Note that another (duplicate)
        // property definition is used here - just to show it can be done.
    c4_IntProp pYears ("age");  // On file this is still the "age" field.

    vAddress.InsertAt(2, pName ["Julian Bream"] + pYears [50]);

        // Preceding commits were only included for demonstration purposes.
    storage.Commit();   // Datafile now contains 4 addresses and 1 info rec.

        // To inspect the data file, use the dump utility: "DUMP MYFILE.DAT".
        // It should generate the following output:
        //
        //      myfile.dat: 3 properties
        //        address[name:S,country:S,age:I],info[version:I]
        //
        //       VIEW     1 rows = _:V address:V info:V
        //          0: subview '_'
        //          0: subview 'address'
        //         VIEW     4 rows = name:S country:S age:I
        //            0: 'John Williams' 'UK' 0
        //            1: 'Paco Pena' 'Spain' 44
        //            2: 'Julian Bream' '' 50
        //            3: 'Julien Coco' 'Netherlands' 0
        //          0: subview 'info'
        //         VIEW     1 rows = version:I
        //            0: 100
        //
        // Note: results will differ if this program is run more than once.
        // In this case, the results must be interpreted with extreme care,
        // since the storage open will throw away the third property and
        // second view before adding them back in with the calls to GetAs.
        //
        // Here is what the file should look like after a second run:
        //
        //      myfile.dat: 3 properties
        //        address[name:S,country:S,age:I],info[version:I]
        //
        //       VIEW     1 rows = _:V address:V info:V
        //          0: subview '_'
        //          0: subview 'address'
        //         VIEW     8 rows = name:S country:S age:I
        //            0: 'John Williams' 'UK' 0
        //            1: 'Paco Pena' 'Spain' 44
        //            2: 'Julian Bream' '' 50
        //            3: 'Julian Bream' '' 0
        //            4: 'Julien Coco' 'Netherlands' 0
        //            5: 'John Williams' 'UK' 0
        //            6: 'Paco Pena' 'Spain' 0
        //            7: 'Julien Coco' 'Netherlands' 0
        //          0: subview 'info'
        //         VIEW     1 rows = version:I
        //            0: 100
        //
        // There is a considerable amount of restructuring taking place here!

    return 0;
}
